/** @file   SpawnItems.h
 * @brief   Declaration of SpawnItem classes.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/07/27 20:28:36 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_SPAWNITEMS_H
#define H_WWW_SPAWNITEMS_H

#include "SpawnEvent.h"


namespace WeWantWar {

/** @class  GameObjectSpawnItem
 * @brief   A spawnitem that creates new gameobjects.
 * @author  Tomi Lamminsaari
 */
class GameObjectSpawnItem : public SpawnItem
{
public:
  /** Constructor.
   * @param   aObjectTag        Opening tag of this item. Operates as the
   *                            the main category of the object.
   */
  GameObjectSpawnItem( const std::string& aObjectTag );
  
  /** Destructor.
   */
  virtual ~GameObjectSpawnItem();
  
  /** From SpawnItem
   */
  virtual void Launch();
  
  /** From SpawnItem
   */
  virtual std::string GetLiteral( LiteralId aId ) const;
  
protected:
  /** The opening tag. Needed when data is loaded from stream. */
  std::string iOpeningTag;
  /** The closing tag. Needed when data is loaded from stream. */
  std::string iClosingTag;
};


/** @class  AnimationSpawnItem
 * @brief   A spawnitem that creates new animations.
 * @author  Tomi Lamminsaari
 */
class AnimationSpawnItem : public SpawnItem
{
public:
  /** Constructor.
   */
  AnimationSpawnItem();
  
  /** Destructor.
   */
  virtual ~AnimationSpawnItem();
  
  /** From SpawnItem
   */
  virtual void Launch();
  
  /** From SpawnItem
   */
  virtual std::string GetLiteral( LiteralId aId ) const;
  
protected:
  /** From SpawnItem
   */
  virtual int ReadParameter(std::istream& aIn, const std::string& aParameter);
};


/** @class  BonusObjectSpawnItem
 * @brief   A spawnitem that creates new bonus items.
 * @author  Tomi Lamminsaari
 */
class BonusObjectSpawnItem : public SpawnItem
{
public:
  /** Constructor.
   */
  BonusObjectSpawnItem();
  
  /** Destructor.
   */
  virtual ~BonusObjectSpawnItem();
  
  /** From SpawnItem
   */
  virtual void Launch();
  
  /** From SpawnItem
   */
  virtual std::string GetLiteral( LiteralId aId ) const;
};


/** @class  SoundSpawnItem
 * @brief   A spawnitem that creates new soundsources.
 * @author  Tomi Lamminsaari
 */
class SoundSpawnItem : public SpawnItem
{
public:
  /** Constructor.
   */
  SoundSpawnItem();
  
  /** Destructor.
   */
  virtual ~SoundSpawnItem();
  
  /** From SpawnItem
   */
  virtual void Launch();
  
  /** From SpawnItem
   */
  virtual std::string GetLiteral( LiteralId aId ) const;
};


/** @class  ParticleSpawnItem
 * @brief   A spawnitem that creates particles
 * @author  Tomi Lamminsaari
 */
class ParticleSpawnItem : public SpawnItem
{
public:
  /** Constructor.
   */
  ParticleSpawnItem();
  
  /** Destructor.
   */
  virtual ~ParticleSpawnItem();
  
  /** From SpawnItem
   */
  virtual void Launch();
  
  /** From SpawnItem
   */
  virtual std::string GetLiteral( LiteralId aId ) const;
  
protected:
  /** From SpawnItem
   */
  virtual int ReadParameter(std::istream& aIn, const std::string& aParameter);
};


/** @class  LightFenceSpawnItem
 * @brief   A spawnitem that creates the lightfences.
 * @author  Tomi Lamminsaari
 */
class LightFenceSpawnItem : public SpawnItem
{
public:
  /** Constructor.
   */
  LightFenceSpawnItem();
  
  /** Destructor.
   */
  virtual ~LightFenceSpawnItem();
  
  /** From SpawnItem
   */
  virtual void Launch();
  
  /** From SpawnItem
   */
  virtual std::string GetLiteral( LiteralId aId ) const;
protected:
  
  /** From SpawnItem
   */
  virtual int ReadParameter( std::istream& aIn, const std::string& aParameter );
};


/** @class  TeleporterSpawnItem
 * @brief   A spawnitem that creates teleporters.
 * @author  Tomi Lamminsaari
 */
class TeleporterSpawnItem : public SpawnItem
{
public:
  /** Constructor.
   */
  TeleporterSpawnItem();
  
  /** Destructor.
   */
  virtual ~TeleporterSpawnItem();
  
  /** From SpawnItem
   */
  virtual void Launch();
  
  /** From SpawnItem
   */
  virtual std::string GetLiteral( LiteralId aId ) const;
protected:
  
  /** From SpawnItem
   */
  virtual int ReadParameter( std::istream& aIn, const std::string& aParameter );
};


/** @class  SpawnItemFactory
 * @brief   Class that provides a static factory method for spawnitems.
 * @author  Tomi Lamminsaari
 */
class SpawnItemFactory
{
public:
  /** Based on given tag this method creates correct spawnitem for it. For
   * instance if @c aTag contains string "&lt;alien&gt;" this method creates an
   * instance of GameObjectSpawnItem. If @c aTag contains string
   * "&lt;bonusitem&gt; then an instance of BonusObjectSpawnItem will be
   * returned.
   * @param   aTag              The tag from level description file that defines
   *                            the type of the spawnitem that should be created.
   * @return  Pointer to new spawnitem. Ownerhip is trasferred.
   */
  static SpawnItem* CreateSpawnItem( const std::string& aTag );
};

};  // end of namespace

#endif
